// Copyright 2009, 2010, 2012, 2015, 2016, 2017, 2020, 2023 Global Virtual Airlines Group. All Rights Reserved.
package org.deltava.util.flightplan;

import java.util.Collection;

import org.deltava.beans.Simulator;
import org.deltava.beans.navdata.*;
import org.deltava.beans.schedule.*;

/**
 * An abstract class to describe flight plan generators for simulators and payware aircraft. 
 * @author Luke
 * @version 10.6
 * @since 2.4
 */

public abstract class FlightPlanGenerator {

	/**
	 * The airline.
	 */
	protected Airline _al;
	
	/**
	 * The departure airport.
	 */
	protected Airport _aD;
	
	/**
	 * The arrivla airport.
	 */
	protected Airport _aA;
	
	/**
	 * The cruise altitude.
	 */
	protected String _altitude;
	
	/**
	 * The standard departure route.
	 */
	protected TerminalRoute _sid;
	
	/**
	 * The standard terminal arrival route.
	 */
	protected TerminalRoute _star;
	
	/**
	 * The route waypoints as rendered in the flight plan for ACARS.
	 */
	protected String _route;
	
	/**
	 * The AIRAC cycle used to generate the flight plan.
	 */
	protected String _navCycle;
	
	/**
	 * Sets the Airline for the flight.
	 * @param al the Airline
	 */
	public void setAirline(Airline al) {
		_al = al;
	}
	
	/**
	 * Sets the departure and arrival Airports for the flight.
	 * @param aD the departure Airport
	 * @param aA the arrival Airport
	 */
	public void setAirports(Airport aD, Airport aA) {
		_aD = aD;
		_aA = aA;
	}
	
	/**
	 * Sets the cruise altitude for the flight.
	 * @param alt the altitude
	 */
	public void setCruiseAltitude(String alt) {
		_altitude = alt;
	}
	
	/**
	 * Updates the deprature route.
	 * @param sid the departure route bean
	 */
	public void setSID(TerminalRoute sid) {
		_sid = sid;
	}
	
	/**
	 * Updates the arrival route.
	 * @param star the arrival route bean
	 */
	public void setSTAR(TerminalRoute star) {
		_star = star;
	}
	
	/**
	 * Sets the route text.
	 * @param route the route text
	 */
	public void setRoute(String route) {
		_route = route;
	}
	
	/**
	 * Updates the AIRAC cycle used to generate this Flight Plan. 
	 * @param cycle
	 */
	public void setNavCycle(String cycle) {
		_navCycle = cycle;
	}

	/**
	 * Generates a flight plan between two airports.
	 * @param waypoints a Collection of waypoints
	 * @return the generated flight plan file
	 */
	public abstract String generate(Collection<NavigationDataBean> waypoints);
	
	/**
	 * Returns the MIME type of flight plans generated by this Generator.
	 * @return the MIME type
	 */
	@SuppressWarnings("static-method")
	public String getMimeType() {
		return "text/plain";
	}
	
	/**
	 * Returns the file extension of flight plans generated by this Generator.
	 * @return the file extension
	 */
	@SuppressWarnings("static-method")
	public String getExtension() {
		return "pln";
	}
	
	/**
	 * Returns the character encoding for flight plans generated by this Generator.
	 * @return the encoding
	 */
	@SuppressWarnings("static-method")
	public String getEncoding() {
		return "utf-8";
	}
	
	/**
	 * Returns the appropriate FlightPlanGenerator for a particular Simulator.
	 * @param sim the Simulator
	 * @return a FlightPlanGenerator
	 */
	public static FlightPlanGenerator create(Simulator sim) {
		switch (sim) {
			case XP9, XP10:
				return new XP9Generator();
			
			case XP11, XP12:
				return new XP11Generator();
			
			case P3D, P3Dv4:
				return new P3DGenerator();
			
			case FS2020:
				return new FS2020Generator();
	
			case FSX:
				return new FSXGenerator();
				
			default:
				return new FS9Generator();
		}
	}
}