// Copyright 2006, 2007, 2009, 2010, 2016, 2018, 2022, 2023 Global Virtual Airlines Group. All Rights Reserved.
package org.deltava.tasks;

import java.util.*;
import java.util.stream.Collectors;
import java.time.*;
import java.sql.Connection;

import org.deltava.beans.flight.*;

import org.deltava.dao.*;
import org.deltava.taskman.*;

import org.deltava.util.system.SystemData;

/**
 * A Scheduled Task to purge Draft Flight Reports from the database.
 * @author Luke
 * @version 11.1
 * @since 1.0
 */

public class DraftPIREPPurgeTask extends Task {

	/**
	 * Initializes the Task.
	 */
	public DraftPIREPPurgeTask() {
		super("Draft PIREP Purge", DraftPIREPPurgeTask.class);
	}

	@Override
	protected void execute(TaskContext ctx) {

		// Determine how many days to purge
		int purgeDays = SystemData.getInt("users.pirep.draft_purge", 30);
		Instant pd = ZonedDateTime.now(ZoneOffset.UTC).minusDays(purgeDays).toInstant();
		log.warn("Purging draft Flight Reports before {}", pd);

		try {
			Connection con = ctx.getConnection();
			
			// Get the DAO and the Flight Reports - remove based on date
			GetFlightReports dao = new GetFlightReports(con);
			Collection<FlightReport> pireps = dao.getByStatus(Collections.singleton(FlightStatus.DRAFT), null).stream().filter(fr -> fr.getDate().isBefore(pd)).collect(Collectors.toList());

			// Get the write DAO and purge
			SetFlightReport wdao = new SetFlightReport(con);
			for (FlightReport fr : pireps) {
				log.warn("Deleting flight {}, Date = {}", fr.getFlightCode(), fr.getDate());
				wdao.delete(fr.getID());
			}
		} catch (DAOException de) {
			log.atError().withThrowable(de).log(de.getMessage());
		} finally {
			ctx.release();
		}

		log.info("Processing Complete");
	}
}