// Copyright 2005, 2008, 2009, 2010, 2011, 2012 Global Virtual Airline Group. All Rights Reserved.
package org.deltava.taglib;

import java.util.*;

import org.deltava.util.StringUtils;

/**
 * A helper class to generate XML elements with attributes.
 * @author Luke
 * @version 5.0
 * @since 1.0
 */

public class XMLRenderer {
	
	private final String _name;
	private final Map<String, String> _attrs = new LinkedHashMap<String, String>();

	/**
	 * Creates a new XML element, with a lowercase name.
	 * @param name the element nam
	 * @throws NullPointerException if name is null
	 */
	public XMLRenderer(String name) {
		super();
		_name = name.toLowerCase();
	}
	
	/**
	 * Returns the XML element name.
	 * @return the element name
	 */
	public String getName() {
		return _name;
	}
	
	/**
	 * Returns whether the element has a particular attribute.
	 * @param name the attribute name
	 * @return TRUE if the attribute exists, otherwise FALSE
	 */
	public boolean has(String name) {
		return _attrs.containsKey(name.toLowerCase());
	}
	
	/**
	 * Clears all element attributes. 
	 */
	public void clear() {
		_attrs.clear();
	}

	/**
	 * Removes an attribute from the element.
	 * @param name the attribute name
	 */
	public void remove(String name) {
		_attrs.remove(name.toLowerCase());
	}
	
	/**
	 * Sets an element attribute.
	 * @param name the attribute name
	 * @param value the attribute value
	 */
	public void setAttribute(String name, String value) {
		if (!StringUtils.isEmpty(name))
			_attrs.put(name.toLowerCase(), value);
	}
	
	/**
	 * Gets an element attribute.
	 * @param name the attribute name
	 * @return the attribute value, or null if none
	 */
	public String get(String name) {
		return _attrs.get(name.toLowerCase());
	}
	
	/**
	 * Renders the open element tag as XML.
	 * @param completeTag TRUE if the tag should be completed, otherwise FALSE
	 * @param finishTag TRUE if the tag should be finished, otherwise FALSE
	 * @return the element rendered to XML
	 */
	public String open(boolean completeTag, boolean finishTag) {
		StringBuilder buf = new StringBuilder("<").append(_name);

		// Append a space if we have attributes
		if (_attrs.size() > 0)
			buf.append(' ');

		// Loop through the attributes
		for (Iterator<Map.Entry<String, String>> i = _attrs.entrySet().iterator(); i.hasNext();) {
			Map.Entry<String, String> e = i.next();

			// Append the attribute name and value
			buf.append(e.getKey());
			if (e.getValue() != null) {
				buf.append("=\"");
				buf.append(e.getValue());
				buf.append('\"');
			}

			// If there's another attribute, add a space
			if (i.hasNext())
				buf.append(' ');
		}

		// Close the tag if requested
		if (completeTag) {
			if (finishTag) {
				if (!_attrs.isEmpty())
					buf.append(' ');
				
				buf.append('/');
			}
			
			buf.append('>');
		}

		return buf.toString();
	}
	
	/**
	 * Renders the open element tag as XML.
	 * @param completeTag TRUE if the tag should be completed, otherwise FALSE
	 * @return the element rendered to XML
	 */
	public String open(boolean completeTag) {
		return open(completeTag, false);
	}
	
	/**
	 * Renders the close element tag as XML.
	 * @return the close tag
	 */
	public String close() {
		StringBuilder buf = new StringBuilder("</").append(_name);
		return buf.append('>').toString();
	}
}