// Copyright 2008, 2009, 2010, 2012, 2016, 2017, 2020, 2023 Global Virtual Airlines Group. All Rights Reserved.
package org.deltava.dao.http;

import java.util.*;
import java.io.InputStream;
import java.time.Instant;
import java.time.format.DateTimeFormatter;

import org.json.*;

import org.apache.logging.log4j.*;

import org.deltava.beans.navdata.AirportLocation;
import org.deltava.beans.wx.*;

import org.deltava.dao.*;

import org.deltava.util.EnumUtils;
import org.deltava.util.cache.*;

/**
 * Loads weather data from FlightAware.
 * @author Luke
 * @version 11.1
 * @since 2.2
 */

public class GetFAWeather extends FlightAwareDAO {
	
	private static final Logger log = LogManager.getLogger(GetFAWeather.class);
	
	private static final Cache<METAR> _wxCache = CacheManager.get(METAR.class, "FlightAwareMETAR");
	private static final Cache<TAF> _fCache = CacheManager.get(TAF.class, "FlightAwareTAF");

	/**
	 * Loads a weather data bean.
	 * @param t the bean type (TAF/METAR)
	 * @param loc the AirportLocation bean
	 * @return the weather bean, or null if not found
	 * @throws DAOException if an I/O error occurs
	 */
	public WeatherDataBean get(WeatherDataBean.Type t, AirportLocation loc) throws DAOException {
		switch (t) {
			case TAF:
				return getTAF(loc);
			case METAR:
			default:
				return getMETAR(loc);
		}
	}

	/**
	 * Gets the METAR for a particular airport.
	 * @param loc the AirportLocation bean
	 * @return the METAR text, or null if not found
	 * @throws DAOException if an I/O error occurs
	 */
	public METAR getMETAR(AirportLocation loc) throws DAOException {
		if (loc == null)
			return null;
		
		// Check the cache
		METAR result = _wxCache.get(loc.getCode());
		if (result != null)
			return result;

		try {
			init(buildURL(String.format("airports/%s/weather/observations", loc.getCode()), Map.of("temperature_units", "C"))); JSONObject jo = null;
			try (InputStream is = getIn()) {
				jo = new JSONObject(new JSONTokener(is));
			}
					
			// Get the parsed METAR
			JSONArray oa = jo.optJSONArray("observations");
			if ((oa == null) || (oa.length() == 0))
				return null;
			
			JSONObject mo = oa.getJSONObject(0);
			result = new METAR();
			result.setAirport(loc);
			result.setDate(Instant.from(DateTimeFormatter.ISO_DATE_TIME.parse(mo.getString("time"))));
			result.setData(mo.getString("raw_data"));
			result.setTemperature(mo.getInt("temp_air"));
			result.setHumidity(mo.optInt("temp_relhum"));
			result.setDewPoint(mo.optInt("temp_dewpoint"));
			result.setWindDirection(mo.optInt("wind_direction"));
			result.setWindSpeed(mo.optInt("wind_speed"));
			result.setWindGust(mo.optInt("wind_speed_gust"));
			Distance d = EnumUtils.parse(Distance.class, mo.optString("visibility_units", "sm"), Distance.SM);
			result.setVisibility(d.getFeet(mo.optInt("visibility", 9999)), false);
			String bpUnits = mo.optString("pressure_units", "mb");
			double rawPressure = mo.optDouble("pressure", 29.92);
			if ("mb".equalsIgnoreCase(bpUnits))
				rawPressure *= 0.02953;
			
			result.setPressure(rawPressure);
			
			// Parse clouds
			JSONArray ca = mo.getJSONArray("clouds");
			for (int x = 0; x < ca.length(); x++) {
				JSONObject co = ca.getJSONObject(x);
				CloudLayer cl = new CloudLayer(co.optInt("altitude"));
				cl.setThickness(CloudLayer.Amount.valueOf(co.optString("type", "BKN")));
				result.add(cl);
			}
			
			// Now try and parse the raw data
			try {
				METAR rm = MetarParser.parse(result.getData());
				result.setIsAutoGenerated(rm.getIsAutoGenerated());
				result.setIsCorrected(rm.getIsCorrected());
				result.setILS(WeatherUtils.getILS(rm));
				for (RunwayVisualRange rvr : rm.getRVR())
					result.add(rvr);
			} catch (Exception e) {
				log.warn("Error parsing METAR - {}", e.getMessage());
			}
			
            _wxCache.add(result);
            return result;
		} catch (Exception e) {
			throw new DAOException(e);
		}
	}
	
	/**
	 * Gets the TAF for a particular airport.
	 * @param loc the AirportLocation bean
	 * @return the TAF text, or null if not found
	 * @throws DAOException if an I/O error occurs
	 */
	public TAF getTAF(AirportLocation loc) throws DAOException {
		if (loc == null)
			return null;
		
		// Check the cache
		TAF result = _fCache.get(loc.getCode());
		if (result != null)
			return result;

		try {
			init(buildURL(String.format("airports/%s/weather/forecast", loc.getCode()), Collections.emptyMap())); JSONObject jo = null;
			try (InputStream is = getIn()) {
				jo = new JSONObject(new JSONTokener(is));
			}
			
			// Get the parsed TAF
			JSONArray rfa = jo.optJSONArray("raw_forecast");
			if ((rfa == null) || (rfa.length() == 0))
				return null;
			
			StringBuilder buf = new StringBuilder();
			for (int x = 0; x < rfa.length(); x++)
				buf.append(rfa.getString(x)).append('\n');
			
			result = new TAF();
			result.setAirport(loc);
			result.setDate(Instant.from(DateTimeFormatter.ISO_DATE_TIME.parse(jo.getString("time"))));
			result.setData(buf.toString());
			_fCache.add(result);
			return result;
		} catch (Exception e) {
			throw new DAOException(e);
		}
	}
}