// Copyright 2008, 2009, 2011, 2016 Global Virtual Airlines Group. All Rights Reserved.
package org.deltava.beans.wx;

import java.util.*;

import org.deltava.beans.flight.ILSCategory;

/**
 * A bean to store airport METAR data.
 * @author Luke
 * @version 7.0
 * @since 2.2
 */

public class METAR extends WeatherDataBean {
	
	private int _temp;
	private int _dp;
	private int _rh;
	
	private int _wDir;
	private int _wSpeed;
	private int _wGust;
	
	private double _bp;

	private int _viz;
	private boolean _vizLT;
	
	private boolean _autoGenerated;
	private boolean _corrected;
	
	private String _remark;
	
	private ILSCategory _ils = ILSCategory.NONE;
	
	private SortedSet<CloudLayer> _clouds = new TreeSet<CloudLayer>();
	private Collection<Condition> _conds = new LinkedHashSet<Condition>(); 
	private Collection<RunwayVisualRange> _rvr = new TreeSet<RunwayVisualRange>();
	
	/**
	 * Returns the temperature.
	 * @return the temperature in degrees
	 */
	public int getTemperature() {
		return _temp;
	}
	
	/**
	 * Returns the dewpoint. 
	 * @return the dewpoint in degrees
	 */
	public int getDewPoint() {
		return _dp;
	}
	
	/**
	 * Returns the relative humidity.
	 * @return the relative humidity as a percentage
	 */
	public int getHumidity() {
		return _rh;
	}
	
	/**
	 * Returns the wind direction.
	 * @return the wind direction in degrees
	 */
	public int getWindDirection() {
		return _wDir;
	}
	
	/**
	 * Returns the sustained wind speed.
	 * @return the wind speed in knots
	 */
	public int getWindSpeed() {
		return _wSpeed;
	}
	
	/**
	 * Returns the wind gust speed.
	 * @return the gust speed in knots
	 */
	public int getWindGust() {
		return _wGust;
	}
	
	/**
	 * Returns the visibility.
	 * @return the visibility in feet
	 */
	public int getVisibility() {
		return _viz;
	}
	
	/**
	 * Returns if the visibility is less than the stated amount.
	 * @return TRUE if less than, otherwise FALSE
	 */
	public boolean getVisibilityLessThan() {
		return _vizLT;
	}

	/**
	 * Returns the barometric pressure.
	 * @return the pressure in inches of mercury
	 */
	public double getPressure() {
		return _bp;
	}
	
	/**
	 * Returns if this is an auto-generated METAR.
	 * @return TRUE if automatically generated, otherwise FALSE
	 */
	public boolean getIsAutoGenerated() {
		return _autoGenerated;
	}
	
	/**
	 * Returns if this is a corrected METAR.
	 * @return TRUE if corrected, otherwise FALSE
	 */
	public boolean getIsCorrected() {
		return _corrected;
	}
	
	/**
	 * Returns the Conditions associated with this METAR.
	 * @return a Collection of Condition beans
	 */
	public Collection<Condition> getConditions() {
		return _conds;
	}
	
	/**
	 * Returns the cloud layers associated with this METAR.
	 * @return a Collection of CloudLayer beans
	 */
	public SortedSet<CloudLayer> getClouds() {
		return _clouds;
	}
	
	/**
	 * Returns the Runway Visual Range reports associated with this METAR. 
	 * @return a Collection of RunwayVisualRange beans
	 */
	public Collection<RunwayVisualRange> getRVR() {
		return _rvr;
	}
	
	/**
	 * Returns the METAR remarks.
	 * @return the remarks
	 */
	public String getRemarks() {
		return _remark;
	}
	
	/**
	 * Returns the ILS category.
	 * @return an ILSCategory
	 */
	public ILSCategory getILS() {
		return _ils;
	}
	
	/**
	 * Adds a condition to this METAR.
	 * @param c the Condition
	 */
	public void add(Condition c) {
		_conds.add(c);
	}
	
	/**
	 * Adds a cloud layer to this METAR.
	 * @param cl the CloudLayer
	 */
	public void add(CloudLayer cl) {
		_clouds.add(cl);
	}

	/**
	 * Adds a Runway Visual Range report to this METAR.
	 * @param rvr the RunwayVisualRange
	 */
	public void add(RunwayVisualRange rvr) {
		_rvr.add(rvr);
	}
	
	/**
	 * Updates the temperature.
	 * @param temp the temperature in degrees
	 */
	public void setTemperature(int temp) {
		_temp = temp;
	}
	
	/**
	 * Updates the dew point.
	 * @param dp the dewpoint in degrees
	 */
	public void setDewPoint(int dp) {
		_dp = dp;
	}
	
	/**
	 * Updates the relative humidity.
	 * @param rh the relative humidity percentage
	 */
	public void setHumidity(int rh) {
		_rh = Math.max(0, Math.min(100, rh));
	}
	
	/**
	 * Updates the wind direction.
	 * @param dir the wind direction in degrees
	 */
	public void setWindDirection(int dir) {
		_wDir = Math.max(0, Math.min(dir, 360));
	}
	
	/**
	 * Updates the sustained wind speed.
	 * @param speed the wind speed in knots
	 */
	public void setWindSpeed(int speed) {
		_wSpeed = Math.max(0, speed);
	}
	
	/**
	 * Updates the wind gust speed.
	 * @param speed the gust speed in knots
	 */
	public void setWindGust(int speed) {
		_wGust = Math.max(0, speed);
	}
	
	/**
	 * Updates the barometric pressure.
	 * @param bp the pressure in inches of mercury
	 */
	public void setPressure(double bp) {
		_bp = bp;
	}
	
	/**
	 * Marks this METAR as automatically generated.
	 * @param autoGen TRUE if auto-generated, otherwise FALSE
	 * @see METAR#getIsAutoGenerated()
	 */
	public void setIsAutoGenerated(boolean autoGen) {
		_autoGenerated = autoGen;
	}
	
	/**
	 * Marks this METAR as Corrected.
	 * @param isCorrected TRUE if corrected, otherwise FALSE
	 */
	public void setIsCorrected(boolean isCorrected) {
		_corrected = isCorrected;
	}
	
	/**
	 * Sets the visibility.
	 * @param viz the visibility in feet
	 * @param lt TRUE if less than the stated amount, otherwise FALSE
	 */
	public void setVisibility(int viz, boolean lt) {
		_viz = Math.max(0, viz);
		_vizLT = lt;
	}
	
	/**
	 * Updates the METAR remarks.
	 * @param r the remarks
	 */
	public void setRemarks(String r) {
		_remark = r;
	}
	
	/**
	 * Updates the ILS Category for this position.
	 * @param ilscat the ILSCategory
	 */
	public void setILS(ILSCategory ilscat) {
		if (ilscat != null)
			_ils = ilscat;
	}
	
	@Override
	public String getIconColor() {
		switch (_ils) {
			case CATI:
				return GREEN;
				
			case CATII:
				return BLUE;
					
			case CATIIIa:
				return ORANGE;
				
			case CATIIIb:
				return PURPLE;
				
			case CATIIIc:
				return RED;
		
			case NONE:
			default:
				return WHITE;
		}
	}
	
	@Override
	public Type getType() {
		return Type.METAR;
	}

	@Override
	public int hashCode() {
		return cacheKey().hashCode();
	}
}