/*
 * Copyright (c) 2001 Matthew Feldt. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided the copyright notice above is
 * retained.
 *
 * THIS SOFTWARE IS PROVIDED ''AS IS'' AND WITHOUT ANY EXPRESSED OR
 * IMPLIED WARRANTIES.
 */

/**
 * MetarTest.java
 *
 * @author Matthew Feldt <developer@feldt.com>
 * @version 0.1, 01/21/2005
 */
package com.feldt.metar;

import com.feldt.metar.exceptions.MetarParseException;

import junit.framework.TestCase;

public class MetarTest extends TestCase {

	public void testParse() throws Exception {
	
		Metar metar = Metar.parse(null, null, null);
		assertNull(metar);
		metar = Metar.parse("", null, null);
		assertNull(metar);
		metar = Metar.parse("", "", null);
		assertNotNull(metar);
	
		try {
			metar = Metar.parse("KCLE", "abc", "");
			fail("abc is not a valid Metar token");
		} catch(MetarParseException e) {
			metarAssertions(metar, "");
			// consume failure
		}
		
		try {
			metar = Metar.parse("KCLE", "1 abc", "");
			fail("1 05006KT is not a valid Metar multi-token visibility");
		} catch(MetarParseException e) {
			metarAssertions(metar, "");
			// consume failure
		}
		
		metar = Metar.parse("KCLE", "KCLE", "");
		assertEquals("KCLE", metar.getStation());
		assertEquals("KCLE", metar.getStation().toString());
		metarAssertions(metar, "station");
		
		metar = Metar.parse("KCLE", "COR", "");
		assertTrue(metar.getCorrected().booleanValue());
		metarAssertions(metar, "corrected");
		
		metar = Metar.parse("KCLE", "280251Z", "2001/01/28 02:51");
		assertNotNull(metar.getDate());
		assertEquals("280251Z", metar.getFormattedDate(false));
		// this assertion fails unless your in my time zone...
//		assertEquals("01/27/2001 09:51 PM EST", metar.getFormattedDate(true));
		metarAssertions(metar, "date");
		
		metar = Metar.parse("KCLE", "05006KT", "");
		assertNotNull(metar.getWind());
		metarAssertions(metar, "wind");
		
		metar = Metar.parse("KCLE", "050006KT 050V240", "");
		assertNotNull(metar.getWind());
		metarAssertions(metar, "wind");
		
		metar = Metar.parse("KCLE", "10SM", "");
		assertNotNull(metar.getVisibility());
		metarAssertions(metar, "visibility");
		
		metar = Metar.parse("KCLE", "1 1/2SM", ""); // multi-token visibility
		assertNotNull(metar.getVisibility());
		metarAssertions(metar, "visibility");
		
		metar = Metar.parse("KCLE", "M13/M17", "");
		assertNotNull(metar.getTemperature());
		metarAssertions(metar, "temperature");
		
		metar = Metar.parse("KCLE", "BLSN", "");
		assertEquals(1, metar.getConditions().size());
		metarAssertions(metar, "conditions");
		
		metar = Metar.parse("KCLE", "FEW008", "");
		assertEquals(1, metar.getClouds().size());
		metarAssertions(metar, "clouds");
		
		metar = Metar.parse("KCLE", "A3072", "");
		assertNotNull(metar.getAltimeter());
		metarAssertions(metar, "altimeter");
		
//		KCLE 230251Z 36019G25KT  -SN   BKN023 OVC027 M11/M14 A2999
	}
	
	private void metarAssertions(Metar metar, String notNull) {
		if (notNull != "station") {
			assertNull(metar.getStation());
		}
		if (notNull != "date") {
			assertNull(metar.getDate());
		}
		if (notNull != "wind") {
			assertNull(metar.getWind());
		}
		if (notNull != "visibility") {
			assertNull(metar.getVisibility());
		}
		if (notNull != "conditions") {
			assertEquals(0, metar.getConditions().size());
		}
		if (notNull != "clouds") {
			assertEquals(0, metar.getClouds().size());
		}
		if (notNull != "temperature") {
			assertNull(metar.getTemperature());
		}
		if (notNull != "altimeter") {
			assertNull(metar.getAltimeter());
		}
		if (notNull != "corrected") {
			assertFalse(metar.getCorrected().booleanValue());
		}
	}
}
