/*
 * Decompiled with CFR 0.152.
 */
package org.gvagroup.pool;

import java.io.Serializable;
import java.text.DecimalFormat;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.PriorityBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.gvagroup.pool.ConnectionInfo;
import org.gvagroup.pool.ConnectionMonitor;
import org.gvagroup.pool.ConnectionPoolEntry;
import org.gvagroup.pool.ConnectionPoolException;
import org.gvagroup.pool.ConnectionWrapper;
import org.gvagroup.pool.PoolEntryComparator;
import org.gvagroup.pool.Recycler;
import org.gvagroup.tomcat.SharedWorker;

public abstract class ConnectionPool<T extends AutoCloseable>
implements Serializable,
AutoCloseable,
Recycler<T> {
    private static final long serialVersionUID = 7191633376038046202L;
    private final transient ReentrantReadWriteLock _lock = new ReentrantReadWriteLock(false);
    private final Lock _r = this._lock.readLock();
    private final Lock _w = this._lock.writeLock();
    private transient DecimalFormat MSFMT = new DecimalFormat("0.000");
    protected final transient Logger log;
    private final String _name;
    private int _poolMaxSize = 1;
    private int _maxRequests;
    private final LongAdder _totalRequests = new LongAdder();
    private final LongAdder _expandCount = new LongAdder();
    private final LongAdder _waitCount = new LongAdder();
    private final LongAdder _fullCount = new LongAdder();
    private final LongAdder _errorCount = new LongAdder();
    private boolean _logStack;
    private long _lastPoolFullTime;
    private long _lastValidationTime;
    private volatile long _maxWaitTime;
    private volatile long _maxBorrowTime;
    private final LongAdder _totalWaitTime = new LongAdder();
    private long _fullWaitTime = 250L;
    private long _borrowWaitTime = 5L;
    private final ConnectionMonitor<T> _monitor;
    private final SortedMap<Integer, ConnectionPoolEntry<T>> _cons = new TreeMap<Integer, ConnectionPoolEntry<T>>();
    private final transient BlockingQueue<ConnectionPoolEntry<T>> _idleCons = new PriorityBlockingQueue<ConnectionPoolEntry<T>>(4, new PoolEntryComparator());
    protected final transient Properties _props = new Properties();

    protected ConnectionPool(int maxSize, String name, int monitorInterval, Class<?> logClass) {
        this.log = LogManager.getLogger(logClass);
        this._name = name;
        this._poolMaxSize = maxSize;
        this._monitor = new ConnectionMonitor(this._name, Math.max(1, monitorInterval), this);
        SharedWorker.register(this._monitor);
    }

    protected void setWaitTime(int borrowWait, int fullWait) {
        this._borrowWaitTime = Math.max(0, borrowWait);
        this._fullWaitTime = Math.max(0, fullWait);
    }

    public String getName() {
        return this._name;
    }

    public abstract String getType();

    private int getNextID() {
        return this._cons.isEmpty() ? 1 : this._cons.lastKey() + 1;
    }

    public int getSize() {
        return this._cons.size();
    }

    abstract int getStaleTime();

    public int getMaxSize() {
        return this._poolMaxSize;
    }

    public long getValidations() {
        return this._monitor.getCheckCount();
    }

    public boolean getLogStack() {
        return this._logStack;
    }

    public Instant getLastValidation() {
        return this._lastValidationTime == 0L ? null : Instant.ofEpochMilli(this._lastValidationTime);
    }

    public long getTotalRequests() {
        return this._totalRequests.longValue();
    }

    public long getFullCount() {
        return this._fullCount.longValue();
    }

    public long getErrorCount() {
        return this._errorCount.longValue();
    }

    public long getExpandCount() {
        return this._expandCount.longValue();
    }

    public long getWaitCount() {
        return this._waitCount.longValue();
    }

    public Duration getMaxWaitTime() {
        return Duration.ofNanos(this._maxWaitTime);
    }

    public Duration getMaxBorrowTime() {
        return Duration.ofNanos(this._maxBorrowTime);
    }

    public void resetMaxTimes() {
        this._maxWaitTime = 0L;
        this._maxBorrowTime = 0L;
    }

    public void setCredentials(String user, String pwd) {
        this._props.setProperty("user", user);
        this._props.setProperty("password", pwd);
    }

    public void setMaxRequests(int maxReqs) {
        this._maxRequests = Math.max(0, maxReqs);
    }

    public void setLogStack(boolean doLog) {
        this._logStack = doLog;
    }

    public void setProperties(Map<?, ?> props) {
        this._props.putAll(props);
    }

    protected abstract ConnectionPoolEntry<T> createConnection(int var1) throws Exception;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public T getConnection() throws ConnectionPoolException {
        ZonedDateTime lastUse;
        ConnectionPoolEntry<T> cpe = null;
        try {
            long wt = System.nanoTime();
            cpe = this._idleCons.poll(this._borrowWaitTime, TimeUnit.MILLISECONDS);
            if (cpe != null) {
                wt = System.nanoTime() - wt;
                if (cpe.isActive() && !cpe.inUse()) {
                    long us = TimeUnit.MICROSECONDS.convert(wt, TimeUnit.NANOSECONDS);
                    Object c = cpe.reserve(this._logStack);
                    this.log.log(us > 2500L ? Level.INFO : Level.DEBUG, "{} reserve {} [{}] ({}ms)", (Object)this._name, cpe, (Object)cpe.getUseCount(), (Object)this.MSFMT.format((double)us / 1000.0));
                    this._totalRequests.increment();
                    this._totalWaitTime.add(wt);
                    return c;
                }
                lastUse = ZonedDateTime.ofInstant(Instant.ofEpochMilli(cpe.getLastUseTime()), ZoneOffset.UTC);
                DateTimeFormatter dtfmt = DateTimeFormatter.ofPattern("HH:mm:ss.SSSSSS");
                this.log.warn("{} retrieved idle/used inactive Connection {} - (idle={}, used={}) by {} on {}", (Object)this._name, cpe, (Object)(!cpe.isActive() ? 1 : 0), (Object)cpe.inUse(), (Object)cpe.getLastThreadName(), (Object)dtfmt.format(lastUse));
                this._errorCount.increment();
                cpe = null;
            }
        }
        catch (InterruptedException ie) {
            this.log.warn("Interrupted waiting for Idle");
        }
        try {
            this._w.lock();
            Optional<ConnectionPoolEntry> nextAvailable = this._cons.values().stream().filter(pe -> !pe.inUse()).findFirst();
            if (!nextAvailable.isPresent() && this._cons.size() < this._poolMaxSize) {
                try {
                    cpe = this.createConnection(this.getNextID());
                    cpe.setDynamic(true);
                    this._cons.put(cpe.getID(), cpe);
                }
                catch (Exception e) {
                    throw new ConnectionPoolException(e);
                }
            } else if (nextAvailable.isPresent()) {
                cpe = nextAvailable.get();
                if (!cpe.isActive()) {
                    try {
                        this.log.info("{} reconnecting Connection {}", (Object)this._name, cpe);
                        cpe.connect();
                        this._expandCount.increment();
                    }
                    catch (Exception e) {
                        throw new ConnectionPoolException(e);
                    }
                } else {
                    this._idleCons.remove(cpe);
                }
            }
            if (cpe != null) {
                Object c = cpe.reserve(this._logStack);
                this.log.debug("{} reserve(wl) {} [{}]", (Object)this._name, cpe, (Object)cpe.getUseCount());
                this._totalRequests.increment();
                lastUse = c;
                return (T)lastUse;
            }
        }
        finally {
            this._w.unlock();
        }
        long waitTime = System.nanoTime();
        try {
            cpe = this._idleCons.poll(this._fullWaitTime, TimeUnit.MILLISECONDS);
            waitTime = System.nanoTime() - waitTime;
            if (cpe != null) {
                T c = cpe.reserve(this._logStack);
                this.log.info("{} reserve(w) {} [{}] ({}ms)", (Object)this._name, cpe, (Object)cpe.getUseCount(), (Object)this.MSFMT.format((double)waitTime / 1000000.0));
                this._maxWaitTime = Math.max(this._maxWaitTime, TimeUnit.MICROSECONDS.convert(waitTime, TimeUnit.NANOSECONDS));
                this._waitCount.increment();
                this._totalWaitTime.add(waitTime);
                return c;
            }
        }
        catch (InterruptedException ie) {
            this.log.warn("Interrupted waiting for Connection");
        }
        long now = System.currentTimeMillis();
        if (now - this._lastPoolFullTime > 5000L) {
            try {
                this._r.lock();
                this.log.error("Pool Full, idleCons = {}", this._idleCons);
                for (Map.Entry<Integer, ConnectionPoolEntry<T>> me : this._cons.entrySet()) {
                    cpe = me.getValue();
                    long activeTime = now - cpe.getLastUseTime();
                    this.log.atError().withThrowable((Throwable)cpe.getStackInfo()).log("Connection {} connected = {}, active = {} ({}ms)", (Object)me.getKey(), (Object)cpe.isConnected(), (Object)cpe.isActive(), (Object)activeTime);
                }
            }
            finally {
                this._r.unlock();
            }
        }
        this._lastPoolFullTime = now;
        this._fullCount.increment();
        throw new ConnectionPoolFullException();
    }

    @Override
    public Duration release(T c) {
        return this.release(c, false);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Duration release(T c, boolean isForced) {
        boolean isStale;
        if (c == null) {
            return Duration.ZERO;
        }
        if (!(c instanceof ConnectionWrapper)) {
            this.log.warn("Invalid Connection returned - {}", (Object)c.getClass().getName());
            this._errorCount.increment();
            return Duration.ZERO;
        }
        ConnectionWrapper cw = (ConnectionWrapper)c;
        ConnectionPoolEntry cpe = null;
        try {
            this._r.lock();
            cpe = (ConnectionPoolEntry)this._cons.get(cw.getID());
        }
        finally {
            this._r.unlock();
        }
        if (cpe == null) {
            this.log.warn("Invalid Connection returned - {}", (Object)cw.getID());
            this._errorCount.increment();
            return Duration.ZERO;
        }
        this.log.debug("{} release {} [{}]", (Object)this._name, (Object)cpe, (Object)cpe.getUseCount());
        try {
            cpe.cleanup();
        }
        catch (Exception e) {
            this.log.warn("{} error cleaning up {} - {}", (Object)this._name, c, (Object)e.getMessage());
            this._errorCount.increment();
            this._monitor.execute();
        }
        long ut = cpe.getUseTime();
        Duration useTime = Duration.ofNanos(ut);
        this._maxBorrowTime = Math.max(this._maxBorrowTime, ut);
        if (isForced) {
            this.log.error("{} forced connection close - Connection {}", (Object)this._name, (Object)cpe);
        }
        boolean bl = isStale = useTime.toMillis() > (long)this.getStaleTime();
        if (cpe.isDynamic() && (isForced || isStale)) {
            this.log.atError().withThrowable((Throwable)cpe.getStackInfo()).log("Closed stale dynamic Connection {} after {} ms", (Object)cpe, (Object)useTime.toMillis());
            cpe.close();
            this._errorCount.increment();
            return useTime;
        }
        if (!cpe.isDynamic()) {
            if (isForced || isStale || this._maxRequests > 0 && cpe.getSessionUseCount() > (long)this._maxRequests) {
                this.log.info("{} restarting Connection {} after {}/{} reservations", (Object)this._name, (Object)cpe, (Object)cpe.getSessionUseCount(), (Object)cpe.getUseCount());
                cpe.close();
                try {
                    cpe.connect();
                }
                catch (Exception se) {
                    this.log.atError().withThrowable((Throwable)se).log("{} cannot reconnect Connection {}", (Object)this._name, (Object)cpe);
                    this._errorCount.increment();
                }
            }
            if (cpe.isConnected()) {
                this.addIdle(cpe);
            }
        } else if (cpe.isConnected()) {
            this.addIdle(cpe);
        }
        this.log.debug("{} released {} [{}] - [{}ms]", (Object)this._name, (Object)cpe, (Object)cpe.getUseCount(), (Object)useTime.toMillis());
        return useTime;
    }

    public void connect(int initialSize) throws ConnectionPoolException {
        if (initialSize < 0 || initialSize > this._poolMaxSize) {
            throw new IllegalArgumentException(String.format("Invalid pool size - %d", initialSize));
        }
        this.log.info("Opening {} (size={})", (Object)this._name, (Object)initialSize);
        this.resetMaxTimes();
        try {
            for (int x = 1; x <= initialSize; ++x) {
                ConnectionPoolEntry<T> cpe = this.createConnection(x);
                this._cons.put(cpe.getID(), cpe);
                this._idleCons.add(cpe);
            }
        }
        catch (Exception e) {
            throw new ConnectionPoolException(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        this.log.info("Shutting down pool {}", (Object)this._name);
        this._monitor.stop();
        try {
            this._w.lock();
            Iterator<ConnectionPoolEntry<T>> i = this._cons.values().iterator();
            while (i.hasNext()) {
                ConnectionPoolEntry<T> cpe = i.next();
                if (cpe.inUse()) {
                    try {
                        this.log.warn("Connection {} in use, waiting", cpe);
                        Thread.sleep(50L);
                    }
                    catch (InterruptedException interruptedException) {
                        // empty catch block
                    }
                }
                this.log.log(cpe.inUse() ? Level.WARN : Level.INFO, "Closing {} Connection {}", (Object)this._name, cpe);
                cpe.close();
                i.remove();
            }
        }
        finally {
            this._w.unlock();
            this.log.info("Shut down {}", (Object)this._name);
        }
    }

    private void addIdle(ConnectionPoolEntry<T> cpe) {
        try {
            boolean hasCon;
            this._r.lock();
            if (cpe.inUse()) {
                cpe.free();
            }
            if (hasCon = this._idleCons.contains(cpe)) {
                this.log.warn("{} entry {} [{}] already in Idle list - {}", (Object)this._name, cpe, (Object)cpe.getUseCount(), this._idleCons);
            } else {
                this._idleCons.offer(cpe);
                this.log.debug("{} added to Idle [{}]", cpe, (Object)cpe.getUseCount());
            }
        }
        finally {
            this._r.unlock();
        }
    }

    public Collection<ConnectionInfo> getPoolInfo() {
        try {
            this._r.lock();
            Collection collection = this._cons.values().stream().map(ConnectionInfo::new).collect(Collectors.toList());
            return collection;
        }
        finally {
            this._r.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void validate() {
        try {
            this.log.debug("{} {} Validator started", (Object)this._name, (Object)this.getType());
            this._lastValidationTime = System.currentTimeMillis();
            this._w.lock();
            Collection<ConnectionPoolEntry<T>> entries = this._cons.values();
            Collection freeEntries = entries.stream().filter(cpe -> cpe.isActive() && !cpe.inUse()).collect(Collectors.toList());
            long idleCount = this._idleCons.stream().filter(ConnectionPoolEntry::isActive).count();
            if (freeEntries.size() != this._idleCons.size() || idleCount != (long)this._idleCons.size()) {
                this.log.warn("{} Free = {} / {}, IdleCount = {}, Idle = {} / {}", (Object)this._name, (Object)freeEntries.size(), (Object)freeEntries, (Object)idleCount, (Object)this._idleCons.size(), this._idleCons);
            }
            for (ConnectionPoolEntry<T> cpe2 : entries) {
                boolean isStale;
                Duration useTime = Duration.ofNanos(cpe2.getUseTime());
                boolean bl = isStale = useTime.toMillis() > (long)this.getStaleTime();
                if (isStale && cpe2.isActive()) {
                    long lastActiveInterval = this._lastValidationTime - cpe2.getWrapper().getLastUse();
                    if (useTime.toMillis() - lastActiveInterval > 15000L) {
                        this.log.warn("Connection reserved for {}ms, last activity {}ms ago", (Object)cpe2.getUseTime(), (Object)lastActiveInterval);
                    }
                    boolean bl2 = isStale = lastActiveInterval > (long)this.getStaleTime();
                }
                if (!cpe2.isActive()) {
                    if (cpe2.inUse()) {
                        this.log.warn("Inactive connection {} in use", cpe2);
                        cpe2.close();
                        continue;
                    }
                    this.log.debug("Skipping inactive connection {}", cpe2);
                    continue;
                }
                if (cpe2.inUse() && isStale) {
                    Duration d = this.release((AutoCloseable)((Object)cpe2.getWrapper()), true);
                    this.log.atError().withThrowable((Throwable)cpe2.getStackInfo()).log("{} releasing stale Connection {} after {}ms ({})", (Object)this._name, cpe2, (Object)d.toMillis(), (Object)cpe2.getLastThreadName());
                    continue;
                }
                if (cpe2.isDynamic() && !cpe2.inUse()) {
                    if (isStale) {
                        this.log.atError().withThrowable((Throwable)cpe2.getStackInfo()).log("{} releasing stale dynamic Connection {}", (Object)this._name, cpe2);
                    } else {
                        this.log.info("{} releasing dynamic Connection {}", (Object)this._name, cpe2);
                    }
                    cpe2.close();
                    boolean wasNotIdle = !this._idleCons.remove(cpe2);
                    if (!wasNotIdle) continue;
                    this.log.warn("{} attempted to remove non-idle connection {}", (Object)this._name, cpe2);
                    continue;
                }
                if (cpe2.inUse()) {
                    this.log.info("Connection {} in use ({})", cpe2, (Object)cpe2.getLastThreadName());
                    continue;
                }
                if (cpe2.inUse()) continue;
                boolean isOK = cpe2.checkConnection();
                this.log.log(isOK ? Level.DEBUG : Level.WARN, "Validated Connection {} - {}", cpe2, (Object)(isOK ? "OK" : "FAILED"));
                if (isOK) continue;
                this.log.warn("Reconnecting Connection {}", cpe2);
                cpe2.close();
                if (this._idleCons.remove(cpe2)) {
                    this.log.debug("{} Validator removed {} from Idle list", (Object)this._name, cpe2);
                }
                try {
                    cpe2.connect();
                    this._idleCons.add(cpe2);
                }
                catch (Exception e) {
                    this.log.atError().withThrowable((Throwable)e).log("Error reconnecting {}", cpe2);
                }
            }
        }
        finally {
            this._w.unlock();
            this.log.debug("{} {} Validator completed", (Object)this._name, (Object)this.getType());
        }
    }

    public static class ConnectionPoolFullException
    extends ConnectionPoolException {
        private static final long serialVersionUID = -1618858703712722475L;

        ConnectionPoolFullException() {
            super("Connection Pool Full");
        }
    }
}

